# 
#
# osds_afddriverstate.pm
# 
# Copyright (c) 2013, 2014, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      osds_afddriverstate.pm - Windows OSD component of afddriverstate.
#
#    DESCRIPTION
#        Purpose
#            Report if AFD drivers are installed and/or loaded
#        Usage
#            afddriverstate [supported] [installed] [loaded]
#
#    NOTES
#
# 

package osds_afddriverstate;
require Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(
                 osds_afd_version
                 osds_afd_validate_drivers
                 osds_afd_compatible
                );

use strict;
use Win32;
use File::Spec::Functions;
use acfslib;
use afdlib;
use osds_afdlib;

sub osds_afd_version
{
  my ($arch);

  # lib_osds_afd_supported() has already verified that the machine
  # architecture and OS type are valid.

  ##### First get the Windows OS type

  $arch = Win32::GetArchName();

  ##### Next get the USM label version

  my ($system_root) = $ENV{SYSTEMROOT};
  my ($driver) = "$system_root/system32/drivers/oracleafd.sys";
  my ($str);
  my ($release_version);

  # convert $driver name format into something findstr understands
  $driver =~ s/\//\\\\/g;
  $driver =~ s/:/:\\\\/g;
  # The usm_label_info[] global contains:
  # usm_ade_label_info_make_header.pl: USM BUILD LABEL: USM_MAIN_WINDOWS.X64_100506
  open (FIND, "findstr /C:\"USM BUILD LABEL\" $driver |");
  $str = <FIND>;
  close(FIND);

  # split the string into 2 sections - the stuff before the sub_string
  # of interest and the stuff after.
  #
  # The usm_label_info[] global contains:
  # usm_ade_label_info_make_header.pl: USM BUILD LABEL: USM_MAIN_WINDOWS.X64_100506
  $str =~ /USM BUILD LABEL: (\S+)/;
  my ($usm_vers) = $1;

  # Get the first 50 characters 
  ($usm_vers) = sprintf("%.50s", $usm_vers);

  # We don't want to export to the user the label info so we strip
  # that from the usm_version, leaving only the date plus some possible chaff
  # at the end - which we chop off. So, USM_MAIN_NT_090112 becomes 090112.
  my @str_array = split (/_/, $usm_vers);
  $usm_vers = $str_array[3];
  $usm_vers = sprintf("%.6s", $usm_vers);
  # Get release version from label name
  $release_version = $str_array[1];
  # Check if the label is patched
  if ($release_version =~ /^((\d+\.){3}?\d+)(\.\d*(.*))$/) {
    # It is, strip the patch number
    # 11.2.0.4.0ACFSPSU becomes 11.2.0.4 (ACFSPSU)
    # untouched otherwise
    $release_version = "$1 ($4).";
  }
  ##### Done. Return the results

  # $OS_TYPE (e.g., "Windows Server 2003") has been set previously in
  # lib_osds_afd_supported().
  return "$OS_TYPE ($arch)/$usm_vers/$release_version";
}

sub osds_afd_validate_drivers
{
  if (!lib_afd_supported())
  {
    # not supported
    return USM_NOT_SUPPORTED;
  }
  return USM_SUPPORTED;
}

# osds_afd_compatible
#
# Make sure drivers are kabi compatible.
#
sub osds_afd_compatible
{
  my ($result) = USM_SUPPORTED;

  return $result;
} #end osds_afd_compatible

