# 
# $Header: has/install/crsconfig/oraohasd.pm /main/14 2016/05/18 08:02:30 hkanchar Exp $
#
# oraohasd.pm
# 
# Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      oraohasd.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      Required variables from the config object $CFG:
#      $CFG->params('ORACLE_OWNER')
#      $CFG->SIHA
#      $CFG->SUPERUSER
#      $CFG->platform_family
#      $CFG->HOST
#      $CFG->params('ORA_DBA_GROUP')
#      $CFG->ORA_CRS_HOME
#      $CFG->ASM_STORAGE_USED
#      $CFG->UPGRADE
#
#    MODIFIED   (MM/DD/YY)
#    muhe        05/10/16 - Fix bug 23249325
#    hkanchar    05/09/16 - Bug# 22992467 Add relocate_clients action to ohasd
#                           asm
#    shullur     06/22/15 - For migrating CHM modules to new root script
#                           changes
#    mperezh     04/15/15 - Fix pxy_create for windows after overwrote
#    muhe        04/14/15 - Add interface functions body for upgrade and patch
#    xyuan       03/25/15 - Fix bug 20709602
#    madoming    03/16/15 - Changes for new framework
#    nkorehis    01/22/15 - bug-19016181 add remove_db_dg_dep action
#    mperezh     01/14/15 - Fix 20328118 - Windows pxy_create
#    luoli       12/22/14 - rsc modeling for downgrade/deconfig
#    mperezh     11/19/14 - Add pxy_create action for storage agent
#    minzhu      10/22/14 - handle anchor kill for haip
#    luoli       08/28/14 - Creation
# 

package oraClusterwareComp::oraohasd;

use parent 'oraClusterwareComp';

use strict;
use English;
use Carp;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;

use crsutils;
use s_crsutils;
use oraafd;
use crska;
use oraacfs;

# Steps to configure OHASD
use constant REGISTER_OHASD_SERVICE => 'ohasd_ConfigCurrentNode_step_1';
use constant START_OHASD_SERVICE    => 'ohasd_ConfigCurrentNode_step_2';
use constant CREATE_OHASD_RESOURCES => 'ohasd_ConfigCurrentNode_step_3';

sub new
{
  my $class = shift;
  # Pass the component name into the constructor 
  my $componentName = @_;

  my $self = $class->SUPER::new(@_);
  $self->_initialize();

  return $self;
}

# Initialize OCR
sub _initialize
{
  my $self = shift;

  my $compName = $self->compName;
  trace("Perform initialization tasks before configuring $compName");
}

# Interface to be implemented
#
# Is the component supported based on platform and user input
sub isSupported
{ 
  # Supported on all platforms
  return TRUE;
}

# Which component does this depend on
sub dependsOn
{ 
  # No dependency
  return undef;
}

# Has the component already been configured
sub isConfigured
{
  #TODO
  return FALSE;
}

#
# Methods for fresh install
#

# Configure action on first node
sub configureFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to configure $compName ".
        "on the first node");
  configSteps($stepIndicator);
}

# Configure action on every node other than first node
sub configureNonFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to configure $compName ".
        "on the non-first node");

  configSteps($stepIndicator);
}

# Configure action on first node after the configured stack
# has been started 
sub postConfigFirstNode 
{
  return SUCCESS;
}

# Configure action on other nodes than the first node after
# the configured stack has been started
sub postConfigNonFirstNode
{
  return SUCCESS;
}

#
# Methods for upgrade
#

# The method for global checks related to OHASD component before upgrade
# This method is called only on the first node when the stack is up.
sub preUpgradeCheck
{
  return SUCCESS;
}

# The method for local checks related to OHASD component before upgrading the first node
sub upgradeCheckFirstNode
{
  return SUCCESS;
}

# The method for local checks related to OHASD component before upgrading the middle node
sub upgradeCheckMiddleNode
{
  return SUCCESS;
}

# The method for local checks related to OHASD component before upgrading the last node
sub upgradeCheckLastNode
{
  return SUCCESS;
}

# Upgrade action on first node
sub upgradeFirstNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the first node");

  configSteps($stepIndicator);
}

# Upgrade action on every node other than first and last node
sub upgradeMiddleNode 
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the middle node");

  configSteps($stepIndicator);
}

# Upgrade action on last node
sub upgradeLastNode
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "on the last node");

  configSteps($stepIndicator);
}

# Upgrade action on the first node after the higher version stack
# has been started
sub postUpgradeFirstNode 
{
  return SUCCESS;
}

# Upgrade action on every node other than first and last node
# after the higher version stack has been started 
sub postUpgradeMiddleNode 
{
  return SUCCESS;
}

# Upgrade action on the last node after the higher version stack
# has been started
sub postUpgradeLastNode 
{
  return SUCCESS;
}

# Downgrade action on nodes other than the last node
sub downgradeNonLastNode
{
  # Delete the OHASD service
  if ($CFG->platform_family eq "windows")
  {
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    s_deltService("OracleOHService");
    s_delete_Oracle_Services("OracleAPXService\\+APX", $ORACLE_HOME);
    s_delete_Oracle_Services("OracleASMService\\+ASM", $ORACLE_HOME);
    s_deltService("OracleMGMTDBService-MGMTDB");
    s_register_service('ohasd');
    if (!isOldVersionLT121())
    {
      s_migrateDBServiceSidsDowngrade();
    }
  }
}

# Downgrade action on last node
sub downgradeLastNode
{
  # same operations as nodes other than the last node
  my $self = shift;
  $self->downgradeNonLastNode();
}

# Whether or not the system reboot is neeeed after configuration
sub rebootRequired
{ 
  return FALSE;
}

sub configureSIHA 
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to configure $compName ".
        "in a SHIA environment");

  ConfigSIHASteps($stepIndicator);
}

sub upgradeSIHA
{
  my $self = shift;
  my $stepIndicator = shift;
  my $compName = $self->compName;

  trace("Executing the step [$stepIndicator] to upgrade $compName ".
        "in a SHIA environment");

  ConfigSIHASteps($stepIndicator);
}

sub unregisterOHASDService
{
  s_unregister_service('ohasd');
}

sub cleanupOHASDResources
{
  my $owner = $CFG->params('ORACLE_OWNER');
  my $pusr = $CFG->SUPERUSER;

  if ($CFG->SIHA)
  {
    if ((! is_dev_env()) && ($CFG->platform_family eq "windows"))
    {
      $owner = '';
    }

    configure_hasd('has', $CFG->HOST, $owner, $owner,
                     $CFG->params('ORA_DBA_GROUP'), "delete");
  }
  else
  {
    if ($CFG->platform_family eq "windows")
    {
      my $NT_AUTHORITY = '';
      if (is_dev_env())
      {
        $NT_AUTHORITY = $CFG->params('ORACLE_OWNER');
      }
      $owner = $pusr = $NT_AUTHORITY;
    }

    configure_hasd('crs', $CFG->HOST, $owner, $pusr,
                     $CFG->params('ORA_DBA_GROUP'), "delete");
  }  
}

# How to start OHASD
sub start
{
  return startOHASD();
}

# How to stop OHASD 
sub stop
{
  return SUCCESS;
}

#
# Private methods
#

sub configSteps
{
  my $stepIndicator = shift;
 
  if (REGISTER_OHASD_SERVICE eq $stepIndicator)
  {
    # Register OHASD service with OS
    return register_ohasd();
  }
  elsif (START_OHASD_SERVICE eq $stepIndicator)
  {
    # Start OHASD service
    return startOHASD();
  }
  elsif(CREATE_OHASD_RESOURCES eq $stepIndicator)
  {
    my $owner = $CFG->params('ORACLE_OWNER');
    my $pusr = $CFG->SUPERUSER;
    if ($CFG->platform_family eq "windows")
    {
      my $NT_AUTHORITY = '';
      if (is_dev_env())
      {
        $NT_AUTHORITY = $CFG->params('ORACLE_OWNER');
      }
      $owner = $pusr = $NT_AUTHORITY;
    }

    # Create OHASD resources and dependencies
    configure_hasd('crs', $CFG->HOST, $owner, $pusr,
                     $CFG->params('ORA_DBA_GROUP'));
    return SUCCESS;
  }
  else
  {
    croak "Step indicator out of bounds";
  }
}

sub ConfigSIHASteps
{
  my $stepIndicator = shift;
 
  if (REGISTER_OHASD_SERVICE eq $stepIndicator)
  {
    # Create OHASD service
    return register_ohasd();
  }
  elsif (CREATE_OHASD_RESOURCES eq $stepIndicator)
  {
    my $owner = $CFG->params('ORACLE_OWNER');
    if ((! is_dev_env()) && ($CFG->platform_family eq "windows"))
    {
      $owner = '';
    }
    
    # Create HA resources and dependencies
    return configure_hasd('has', $CFG->HOST, $owner, $owner,
                            $CFG->params('ORA_DBA_GROUP'));
  }
  else
  {
    croak "Step indicator out of bounds";
  }
}

sub register_ohasd
{
   my $rc = s_register_service('ohasd');

   if ($rc == SUCCESS) {
      s_install_initd() or die(dieformat(317));
   }

   return $rc;
}

sub addResourceType
{
  my ($type, $baseType, $templateFile, $ohasdlog) = @_; 

  trace("Registering resource type '$type'");

  my $ORACLE_HOME = $ENV{'ORACLE_HOME'};
  my $CRS_HOME_TEMPLATE = catdir($ORACLE_HOME, "crs", "template");
  my $infile  = catfile($CRS_HOME_TEMPLATE, $templateFile);
  my $outfile = catfile($ohasdlog, $type);
  instantiateTemplate($infile, $outfile);
  
  my $crsctl = crs_exec_path('crsctl');
  my $status = system_cmd($crsctl, "add", "type", $type, "-basetype",
                 $baseType, "-file", "$outfile", "-init");
  if ($status != 0)
  {
    trace("Failed to add type '$type'");
    die(dieformat(270, $type));    
  }
  else
  {
    trace("Type '$type' added successfully");
  }

  unlink($outfile); 
}

sub configure_hasd {
  my $mode     = $_[0];
  my $hostname = $_[1];
  my $owner    = $_[2]; # the owner of the software bits aka CRS USER
  my $pusr     = $_[3]; # the privileged user 
  my $grp      = $_[4];
  my $action   = $_[5];
  my @out      = ();
  my $status;

  # Register these resources for SIHA only
  my @registerTypesHAS = ("cssd", "evm");

  # Register these resources for clusterware only
  my @registerTypesCRS = ("evm", "mdns", "gpnp", "gipc", "cssd", "cssdmonitor",
                          "crs", "ctss", "crf", "asm", "drivers.acfs", "oka");
 
  if ($CFG->platform_family eq "windows") {
     if (! $owner) {
        $pusr = '';
        $grp  = '';
     }
  } else {
     push(@registerTypesHAS, "diskmon");
     push(@registerTypesCRS, "diskmon");
  }
      
  my $ORACLE_HOME = $ENV{'ORACLE_HOME'};

  # Make sure CRS_HOME is set
  if ( ! $ORACLE_HOME ) {
    die(dieformat(268));
  }

  # Set Homes
  my $CRS_HOME_BIN = catdir($ORACLE_HOME,"bin");
  my $crsctl = catfile( $CRS_HOME_BIN, "crsctl");

  my @types;
  if ($mode eq "has") {
    @types = @registerTypesHAS;
  } elsif ($mode eq "crs") {
    @types = @registerTypesCRS;
  }
  
  # Add AFD type if AFD is supported
  if(isAFDSupported())
  {
    push(@types, "driver.afd");
  }
 
  my $ohasdbaseType = 'ora.ohasdbase.type';
  my $daemonType = 'ora.daemon.type';
  my $ocrstorageType = 'ora.storage.type';
  my $haipType = 'ora.haip.type';
  my $file;
  my $name;
  my $logdir = catdir($CFG->ORA_CRS_HOME, "log", $hostname);
  my $ohasdlog = catdir($logdir, "ohasd");

  #set default action to add
  if (! $action) {
     $action = "add";
  }

  if ($action eq "add")
  {
    addResourceType($ohasdbaseType, "cluster_resource", "ohasdbase.type", $ohasdlog);
    addResourceType($daemonType, $ohasdbaseType, "daemon.type", $ohasdlog);

    if ($mode eq "crs")
    {
      addResourceType($ocrstorageType, $ohasdbaseType, "storage.type", $ohasdlog);
    }

    if (isHAIPsupported())
    {
      addResourceType($haipType, "cluster_resource", "haip.type", $ohasdlog);
      # as part of the resource addition, also fixup any system config files
      s_CheckNetworkConfig();
    }
  }

  # register the infrastructure resources
  foreach my $type (@types)
  {
    $file = $type . '.type';
    $name = 'ora.' . $type . '.type';

    if ($action eq "add")
    {
      addResourceType($name, $daemonType, $file, $ohasdlog);
    }
  }

  # acting on resources
  if ($mode eq "crs")
  {
    configure_ohasd_CRS($owner, $pusr, $grp, $action);
  } 
  elsif ($mode eq "has")
  {
    configure_ohasd_SIHA($owner, $pusr, $grp, $action);
  }

  if (($action eq "delete") && isHAIPsupported())
  {
    trace("Unregistering type '$haipType'");
    @out = system_cmd_capture($crsctl, "delete","type", $haipType, "-init");
  }

  if ($action eq "delete")
  {
     trace("Deleting base types");

     foreach my $type (@types)
     {
       $name = 'ora.' . $type . '.type';
       @out =  system_cmd_capture($crsctl, "delete","type", $name, "-init");
     }

     @out = system_cmd_capture($crsctl, "delete","type", $daemonType, "-init");

     if ($mode eq "crs")
     {
       trace("Unregistering type '$ocrstorageType'");
       system_cmd_capture($crsctl, "delete", "type", $ocrstorageType, "-init");
     }

     trace("Unregistering type '$ohasdbaseType'");
     system_cmd_capture($crsctl, "delete", "type", $ohasdbaseType, "-init");
  } 

  return TRUE;
}

sub configure_ohasd_SIHA
{
  my ($owner, $pusr, $grp, $action) = @_;

  # set the owners : user IDs to spawn agents as
  my $AFDOWNER  = $pusr;
  my $CSSOWNER  = $pusr;
  my $EVMOWNER  = $owner;

  # set the users : explit execution rules (may or may not be equal to owner)
  my $AFDUSER  = $owner;
  my $CSSUSER  = $owner;
  my $EVMUSER  = $owner;

  my @css_attr = ("CSS_USER=$CSSUSER");
  my @diskmon_attr = ("USR_ORA_ENV=ORACLE_USER=$CSSUSER");
  my @evm_attr = ("ACL='owner:$EVMOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$EVMUSER:rwx'");
  my $css_startattr = "";
  my $css_stopattr  = "";

  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attr, $ENV{'CSSDAGENT_ATTR'}; }

  # SI-HA cssd does not depend on mdnsd/gpnpd
  push @css_attr, "ACL='owner:$CSSOWNER:rwx,pgrp:$grp:rwx,other::r--'";
  push @css_attr, "RESTART_ATTEMPTS=5";
  push @diskmon_attr, "ACL='owner:$CSSOWNER:rwx,pgrp:$grp:rwx,other::r--'";

  if ($CFG->platform_family ne "windows")
  {
    $css_startattr = "START_DEPENDENCIES='weak(concurrent:ora.diskmon)'";
    $css_stopattr  = "STOP_DEPENDENCIES='hard(shutdown:ora.diskmon)'";

    push @diskmon_attr, "START_DEPENDENCIES='weak(concurrent:ora.cssd)" .
                        "pullup:always(ora.cssd)'";

    crsctlResource($action, "ora.diskmon", "ora.diskmon.type", \@diskmon_attr);
  }

  if(isAFDSupported())
  {
     # START_DEPENDENCY: There is a resource dependency between CSS and AFD as
     # CSS needs AFD disks(ASM disks managed by AFD) for voting files.

     # STOP_DEPENDENCY: There's need for CSS resource to be stopped when 
     # AFD is stopped, hence the shutdown dependency.

     if ($CFG->platform_family eq "windows")
     {
       $css_startattr = "START_DEPENDENCIES='hard(ora.driver.afd)'";
       $css_stopattr  = "STOP_DEPENDENCIES='hard(shutdown:ora.driver.afd)'";
     }  
     else
     { 
       $css_startattr = 
       "START_DEPENDENCIES='weak(concurrent:ora.diskmon)".
       "hard(ora.driver.afd)'";
       $css_stopattr = 
       "STOP_DEPENDENCIES='hard(shutdown:ora.diskmon,shutdown:ora.driver.afd)'";
     }

     # Add AFD resource at the begin
     if($action eq "add")
     {
       if(FAILED == actionAFDDriversResource($AFDOWNER,$AFDUSER,$grp,$action))
       {
          die(dieformat(2503, "ora.driver.afd")); 
       }
     }
  }
   
  push @css_attr, $css_startattr;
  push @css_attr, $css_stopattr;

  crsctlResource($action, "ora.cssd", "ora.cssd.type", \@css_attr);
  crsctlResource($action, "ora.evmd", "ora.evm.type", \@evm_attr);

  # Delete AFD resource at the end
  if(isAFDSupported() &&  ($action eq "delete"))
  {
    actionAFDDriversResource($AFDOWNER, $AFDUSER, $grp, $action);
  }
}

sub configure_ohasd_CRS
{
  my ($owner, $pusr, $grp, $action) = @_;

  # set the owners : user IDs to spawn agents as
  my $AFDOWNER  = $pusr;
  my $MDNSOWNER = $owner;
  my $GPNPOWNER = $owner;
  my $GIPCOWNER = $pusr;
  my $CSSOWNER  = $pusr;
  my $EVMOWNER  = $owner;
  my $CRSOWNER  = $pusr;
  my $OCRSTORAGEOWNER = $pusr;

  # set the users : explit execution rules (may or may not be equal to owner)
  my $AFDUSER  = $owner;
  my $MDNSUSER = $owner;
  my $GPNPUSER = $owner;
  my $GIPCUSER = $owner;
  my $CSSUSER  = $owner;
  my $EVMUSER  = $owner;
  my $CRSDUSER = $owner;
  my $OCRSTORAGEUSER = $owner;
  my $CSSUSERSTRING  = "user:".$CSSUSER;
  my $OCRSTSTRING    = "user:".$OCRSTORAGEOWNER;
  my $GRPSTRING      = "group:".$grp;
  my $STORAGEACTIONACL = $OCRSTSTRING.",".$GRPSTRING; 

  my @evm_attr = ("ACL='owner:$EVMOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$EVMUSER:rwx'");
  my @asm_attr = ("ACL='owner:$CSSUSER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$CSSUSER:rwx'");
  my @ocrstorage_attr = ("ACL='owner:$OCRSTORAGEOWNER:rw-,pgrp:$grp:r--," .
                              "other::r--,user:$OCRSTORAGEUSER:r-x'");
  my @css_attr    = ("CSS_USER=$CSSUSER");
  my @crsd_attr   = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-," .
                          "other::r--,user:$CRSDUSER:r-x'");
  my @css_attrmon = ("CSS_USER=$CSSUSER",
                     "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                          "other::r--,user:$CSSUSER:r-x'");
  my @diskmon_attr = ("USR_ORA_ENV=ORACLE_USER=$CSSUSER");
  my @gipc_attr = ("ORACLE_USER=$GIPCUSER");
  my $css_stopattr = ""; 

  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attr, $ENV{'CSSDAGENT_ATTR'}; }
  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attrmon, $ENV{'CSSDAGENT_ATTR'}; }
  
  push @css_attr, "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$CSSUSER:r-x'"; 
  push @css_attr, "AUTO_START=always";
  if (($CFG->platform_family eq "windows") && ($CSSUSER eq ""))
  {
    $CSSUSERSTRING = "hapriv:";
  }
  push @asm_attr, "ACTIONS='CRSDASM_START,$CSSUSERSTRING ".
                   "relocate_clients,$CSSUSERSTRING'";

  if ($CFG->platform_family eq "windows")
  {
    if (is_dev_env())
    {
       if ($grp eq "")
       {
         push @ocrstorage_attr, "ACTIONS='set_usr_cred,user:$OCRSTORAGEOWNER ".
                  "vol_create,user:$CSSUSER "."pxy_create,user:$CSSUSER ".
                  "vol_delete,user:$CSSUSER "."remove_db_dg_dep,user:$CSSUSER'";
       }
    }
    else
    {
      if ($grp eq "")
      {
        $GRPSTRING = "group:ORA_INSTALL";
      }
      if ($OCRSTORAGEOWNER eq "")
      {
        $OCRSTSTRING = "hapriv:";
      }
      if (($grp eq "") && ($OCRSTORAGEOWNER eq ""))
      {
        $STORAGEACTIONACL = $GRPSTRING;
      }
      else
      {
        $STORAGEACTIONACL = $OCRSTSTRING.",".$GRPSTRING;
      }
      if ($CSSUSER eq "")
      {
        $CSSUSERSTRING = "hapriv:";
      }
      # Action string will be formed to look like:
      #  ACTIONS='set_usr_cred,group:ORA_INSTALL vol_create,hapriv: 
      #           pxy_create,hapriv: vol_delete,hapriv:'
      # 'hapriv:' is required by clusterware to process
      # the user and fill according each platform.
      push @ocrstorage_attr, "ACTIONS='set_usr_cred,$STORAGEACTIONACL ".
               "vol_create,$CSSUSERSTRING "."pxy_create,$CSSUSERSTRING ".
               "vol_delete,$CSSUSERSTRING "."remove_db_dg_dep,$CSSUSERSTRING'";
    }
  }
  else
  {
     push @ocrstorage_attr, 
               "ACTIONS='set_usr_cred,user:$OCRSTORAGEOWNER,group:$grp ".
               "vol_create,user:$CSSUSER "."pxy_create,user:$CSSUSER ".
               "vol_delete,user:$CSSUSER "."remove_db_dg_dep,user:$CSSUSER'";
  }

  if ($CFG->platform_family eq "windows")
  {
    push @css_attr, "START_DEPENDENCIES='hard(ora.gpnpd,ora.gipcd,ora.cssdmonitor)" .
                                        "pullup(ora.gpnpd,ora.gipcd)'";
    $css_stopattr = "STOP_DEPENDENCIES='hard(intermediate:ora.gipcd,intermediate:ora.cssdmonitor";
  }
  else
  {
    push @css_attr, "START_DEPENDENCIES='weak(concurrent:ora.diskmon)" .
                                 "hard(ora.cssdmonitor,ora.gpnpd,ora.gipcd)" .
                                 "pullup(ora.gpnpd,ora.gipcd)'";
    $css_stopattr = "STOP_DEPENDENCIES='hard(intermediate:ora.gipcd,shutdown:ora.diskmon,intermediate:ora.cssdmonitor";

    push @diskmon_attr, "START_DEPENDENCIES='weak(concurrent:ora.cssd)" .
                                            "pullup:always(ora.cssd)'";
  }

  if(isAFDSupported())
  {
    # There is a resource dependency between CSS and AFD as CSS needs AFD 
    # disks(ASM disks managed by AFD) for voting files.
    # cssdmonitor is the lowest resource in the cssd dependency tree 
    # (having hard dependency) in the css group. So, we are setting 
    # dependency between cssdmonitor and afd. 
    push @css_attrmon, "START_DEPENDENCIES='hard(ora.driver.afd)'" ;
      
    # There's need for CSS resource to be stopped when AFD is stopped,
    # hence the shutdown dependency.
    push @css_attr, $css_stopattr.",shutdown:ora.driver.afd)'";

  }
  else
  {
    push @css_attr, $css_stopattr.")'";
  }

  # Add AFD resource at the begin
  if($action eq "add")
  {
    if(FAILED == actionAFDDriversResource($AFDOWNER, $AFDUSER, $grp, $action))
    {
       die(dieformat(2503, "ora.driver.afd")); 
    }
  }

  push @diskmon_attr, "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                           "other::r--,user:$CSSUSER:r-x'";

  my @mdns_attr = ("ACL='owner:$MDNSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$MDNSUSER:rwx'");
  crsctlResource($action, "ora.mdnsd", "ora.mdns.type", \@mdns_attr);

  my @gpnp_attr = ("ACL='owner:$GPNPOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$GPNPUSER:rwx'");
  crsctlResource($action, "ora.gpnpd", "ora.gpnp.type", \@gpnp_attr, "-f");

  push @gipc_attr, "ACL='owner:$GIPCOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$GIPCUSER:rwx'";
  crsctlResource($action, "ora.gipcd", "ora.gipc.type", \@gipc_attr);

  if ($CFG->platform_family ne "windows") 
  {
    crsctlResource($action, "ora.diskmon", "ora.diskmon.type", \@diskmon_attr);
  }
  
  crsctlResource($action, "ora.cssdmonitor", "ora.cssdmonitor.type", \@css_attrmon, "-f");
  crsctlResource($action, "ora.cssd", "ora.cssd.type", \@css_attr); 
  crsctlResource($action, "ora.evmd", "ora.evm.type", \@evm_attr);

  my @ctss_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
  crsctlResource($action, "ora.ctssd", "ora.ctss.type", \@ctss_attr);

  if (isHAIPsupported())
  {
    my @haip_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
    
    isHAIPNonFatal()? push @haip_attr, "USR_ORA_HA_FATAL=0":push @haip_attr, "USR_ORA_HA_FATAL=1";

    crsctlResource($action, "ora.cluster_interconnect.haip", "ora.haip.type", \@haip_attr);
  }

  if (isCHMSupported())
  {
    my @crf_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
    crsctlResource($action, "ora.crf", "ora.crf.type", \@crf_attr);
  }

  # ora.ctssd dependency only needed for cluster and not for siha
  if (isHAIPsupported() )
  {
    if( ($OSNAME eq "hpux") || ($OSNAME eq "aix") )
    {
        push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.ctssd)" .
          "pullup(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
            "weak(ora.cluster_interconnect.haip,ora.drivers.acfs)'";
    }
    else
    {
        push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
          "pullup(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
            "weak(ora.drivers.acfs)'";
    }
    push @asm_attr, "STOP_DEPENDENCIES='hard(intermediate:ora.cssd," .
                                    "shutdown:ora.cluster_interconnect.haip)'";
  }
  else
  {
    push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.ctssd)" .
                                        "pullup(ora.cssd,ora.ctssd)" .
                                        "weak(ora.drivers.acfs)'";
  }

  my $crsctl = crs_exec_path('crsctl');
  my $rc;

  if ($action eq "add")
  {
    $rc = system_cmd($crsctl, 'add', 'category', 'ora.hub.category', '-init',
                         '-attr', '"ACTIVE_CSS_ROLE=hub"');
  }

  if ($rc != 0)
  {
    trace("Failed to '$action' ora.hub.category");
  }

  # When OCR is on ASM, add ora.asm as a HARD and PULLUP start dependency
  # These need to be consistent with :
  # has/crs/template/crs.type
  # prou.c
  if (($CFG->ASM_STORAGE_USED) ||
       ($CFG->UPGRADE && (1 == $CFG->oldconfig('ASM_CONFIGURED'))))
  {
    # These dependencies are used for ASM server cluster install, and
    # upgrade from both standard and flex ASM.
    # For ASM client cluster, the default dependencies will be used for
    # both install and upgrade.
    # For app cluster install, the default dependencies will be used.
    if (! isFarASM())
    {
      push @ocrstorage_attr, "START_DEPENDENCIES='hard(ora.cssd)weak(ora.asm)pullup(ora.asm,ora.cssd)pullup(ora.ctssd)'";
      push @ocrstorage_attr, "STOP_DEPENDENCIES='hard(shutdown:ora.asm,intermediate:ora.cssd)'";
    }
  }

  crsctlResource($action, "ora.asm", "ora.asm.type", \@asm_attr);
  crsctlResource($action, "ora.storage", "ora.storage.type", \@ocrstorage_attr);
  crsctlResource($action, "ora.crsd", "ora.crs.type", \@crsd_attr);

  actionOKADriversResource($action);
  actionOKADriversResource("adddependency", "weak", "start", "ora.crsd");

  # Unregister 'ora.hub.category' after deleting ora.asm as it is referenced by resource 'ora.asm'
  if ($action eq "delete")
  {
    system_cmd_capture($crsctl, 'delete', 'category', 'ora.hub.category', '-init');
  }


  # add or delete the ACFS resource
  my $oraacfs = $CFG->compACFS;
  if ( ($action eq "add") &&
       (FAILED == $oraacfs->configureFirstNode($oraacfs->CREATE_ACFS_RESOURCE)))
  { 
    print_error(425);
  }
  elsif ($action eq "delete")
  {
    $oraacfs->configureFirstNode($oraacfs->REMOVE_ACFS_RESOURCE);
  }

  # Delete AFD resource at the end
  if( isAFDSupported() && ($action eq "delete") )
  {
     actionAFDDriversResource($AFDOWNER, $AFDUSER, $grp, $action);
  }
}

#----------------------( instantiateTemplate )--------------------------#
#                                                                       #
#                                                                       #
#  FUNCTION: instantiateTemplate                                        #
#                                                                       #
#  PURPOSE: Instantiates the cap file with the CRS HOME location        #
#                                                                       #
#-----------------------------------------------------------------------#
sub instantiateTemplate
{
  my $ORACLE_HOME = $CFG->params('ORACLE_HOME');

  my ($inFile, $outFile) = @_;

  #TODO Define this based on platforms
  my $FSEP = '/';

  # If I can read the template or cap, instantiate the file replacing any 
  # special values
  if ( -r $inFile) 
  {
    open (INF, "<", "$inFile") or
        fatal("Unable to open $inFile, $!, ");

    # Make sure to open output file safely
    if ( -r $outFile ) 
    {
      trace("Removing pre existing $outFile from a previous run.");
      unlink($outFile) or 
          die(dieformat(168, $outFile, $!));
    }
    open (OUTF, ">", "$outFile") or
        die(dieformat(207, $outFile, $!));

    # 
    # Filter Transformations
    #
    while (<INF>) 
    {
      # Modify CRS Home
      s/%ORA_CRS_HOME%/$ORACLE_HOME/;

      # Modify File Separators for NT
      s/\/\//$FSEP/g;
      print OUTF $_;
    }
    close(INF) or
        die(dieformat(272, $inFile, $!));

    close(OUTF) or
        die(dieformat(272, $outFile, $!));

  } 
  else 
  {
    print_error(32, $inFile);
    trace("$inFile is not readable.",
          "Verify the value of ORACLE_HOME, ");
    die(dieformat(273, $inFile));
  }

}

sub startOHASD
{
   trace ("Starting ohasd");
   my $srv = 'ohasd';
   my $rc = SUCCESS;

   # Check if the service/daemon has started
   trace ("Checking the status of $srv");
   my $ohasdStatus = check_service ($srv, 3, FALSE);

   if ($ohasdStatus) {
      trace ("$srv is already running");
   }
   else {
      trace ("$srv is not already running.. will start it now");
      start_service($srv);
   }

   # Check if the service/daemon has started
   trace ("Checking the status of $srv");
   $ohasdStatus = check_service ($srv, 360);

   if ($ohasdStatus) {
      trace ("$srv started successfully");
   }
   else {
      trace ("$srv has failed to start");
      $rc = FAILED;
   }

   return $rc;
}


1;

