# 
# $Header: has/install/crsconfig/oraios.pm /st_has_12.2.0.1.0/1 2016/08/10 08:58:35 samjo Exp $
#
# oraios.pm
# 
# Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      oraios.pm - Perl module for cofiguring ASM I/O Server (IOS) 
#
#    DESCRIPTION
#      Perl module for cofiguring ASM I/O Server (IOS) 
#
#    NOTES
#      Required variables from the config objecj $CFG:
#      $CFG->ASM_STORAGE_USED
#      $CFG->oldconfig('ASM_CONFIGURED')
#      $CFG->oldconfig('LOCAL_NODE_NUM')
#      $CFG->oldconfig('ORA_CRS_VERSION')
#      $CFG->oldconfig('ORA_CRS_HOME')
#      $CFG->params('ORATAB_LOC')
#      $CFG->params('ORACLE_OWNER')
#      $CFG->params('ORACLE_BASE')
#      $CFG->ORA_CRS_HOME
#      $CFG->oldconfig('NODE_CONFIG_ROLE')
#      $CFG->OLD_CRS_HOME
#
#    MODIFIED   (MM/DD/YY)
#    samjo       07/28/16 - Fix bug 24357370
#    bbeelamk    02/10/16 - Fix bug 22642839
#    samjo       11/05/15 - Fix bug 22115751
#    bbeelamk    09/30/15 - Fix bug 20679385
#    bbeelamk    09/15/15 - Changing srvctl method
#    samjo       09/24/q4 - Adding IOS configuration
#    xyuan       09/23/14 - Creation
# 
package oraClusterwareComp::oraios;

use parent 'oraClusterwareComp';

use strict;
use English;
use Carp;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Env qw(SRVM_TRACE);

use crsutils;
use s_crsutils;

# Steps to configure ASM on the first node
use constant IMPORT_ASM_CREDS_FARASM  => 'asm_ConfigFirstNode_step_1';
use constant CONFIGURE_ASM            => 'asm_ConfigFirstNode_step_2';

# Steps to upgrade ASM on the first node
use constant UPGRADE_ASM              => 'asm_UpgradeFirstNode_step_1';

sub new
{
  my $class = shift;
  # Pass the component name into the constructor 
  my $componentName = @_;

  my $self = $class->SUPER::new(@_);
  $self->_initialize();

  return $self;
}

# Initialization
sub _initialize
{
  my $self = shift;

  my $compName = $self->compName; 
  trace("Perform initialization tasks before configuring $compName");
}

#
# Each specific component, which inherits from this class, can 
# reimplement (override) the following methods:

# Is the component supported based on platform and user input
#------------------------------------------------------------------------------
# Function: Find if IOS is supported
# Args    : None
# Returns : TRUE or FALSE
# Notes   : IOS gets configured only in a Domain Services Cluster
#           IOS supported on Linux and Solaris in 12.2.0.1.0
#        
#------------------------------------------------------------------------------
sub isSupported
{
  my $ret = FALSE;

  # Not supported in Member Cluster.
  if (isDSCConfigured()) 
  {
    if (($^O =~ /linux/i) || ($^O =~ /solaris/i))
    {
      $ret = TRUE;
    }
  }
  
  return $ret;
}

# Which component does this depend on
sub dependsOn 
{ 
  my @dependency;

  push @dependency, "ASM";
  return @dependency; 
}

# Has the component already been configured
#------------------------------------------------------------------------------
# Function: Find if IOS resource is already configured
# Args    : None
# Returns : TRUE or FALSE
# Notes   : 
#------------------------------------------------------------------------------
sub isConfigured
{
  my @output;
  my $run_as_owner = TRUE;
  my $ret;
  my $status;

  trace("Invoking srvctl config ioserver");

  $status = srvctl($run_as_owner, "config ioserver");
  if ($status == TRUE) 
  {
    trace ("'srvctl config ioserver' ... success");
    $ret = TRUE;
  } 
  else 
  {
    trace ("'srvctl config ioserver' error. Not configured.");
    $ret = FALSE;
  }

  return $ret;
}

#
# Methods for install
#


# Configure actions on first node
sub configureFirstNode 
{
  # NOOP for IOS
  return SUCCESS;
}

# Configure actions on every node other than first node
sub configureNonFirstNode 
{
  # NOOP for IOS
  return SUCCESS;
}

# Configure action on first node after the configured stack
# has been started
sub postConfigFirstNode
{
  my $self = shift;

  trace("Start to configure IOS on the first node ...");
  return configureIOS();
}

# Configure action on other nodes than the first node after
# the configured stack has been started
sub postConfigNonFirstNode
{
  # NOOP for IOS
  return SUCCESS;
}

#
# Methods for upgrade
#

# Upgrade action on first node
sub upgradeFirstNode
{
  # NOOP for IOS
  return SUCCESS;
}

# Upgrade action on every node other than first and last node
sub upgradeMiddleNode 
{
  # NOOP for IOS
  return SUCCESS;
}

# Upgrade action on last node
sub upgradeLastNode
{
  # NOOP for IOS
  return SUCCESS;
}

# Upgrade action on the first node after the higher version stack
# has been started
sub postUpgradeFirstNode
{
  return SUCCESS;
}

# Upgrade action on every node other than first and last node
# after the higher version stack has been started
sub postUpgradeMiddleNode
{
  return SUCCESS;
}

# Upgrade action on the last node after the higher version stack
# has been started
sub postUpgradeLastNode
{
  my $self = shift;

  trace("Start to configure IOS on the last node after upgrade ...");
  return configureIOS();
}

# Whether or not the system reboot is needed after ASM is configured
sub rebootRequired { return FALSE; }

# How to start the component
sub start 
{
  my $run_as_oracle_owner = SUCCESS;
  my $status = srvctl($run_as_oracle_owner, "start ioserver");

  if ($status) {
    trace ("start ios ... success");
  }
  else {
    print_error(113);
    return FAILED;
  }

  return SUCCESS;
}

# How to stop the component
sub stop 
{ 
  my $run_as_oracle_owner = SUCCESS;
  my $status = srvctl($run_as_oracle_owner, "stop ioserver -f");

  if ($status) {
    trace ("stop ioserver ... success");
  }
  else {
    print_error(114);
    return FAILED;
  }

  return SUCCESS; 
}


#
# Private methods
#

# private methods called by above APIs

# configure IOS on the first node during fresh install



=head2 configureIOS

   Configures IOS

=head3 Parameters

   None

=head3 Returns

  TRUE  - IOS configuration was     created or updated
  FALSE - IOS configuration was not created or updated

=head3 Notes

  This will configure IOS as part of the configuration if it is successful
  This issues : srvctl add ioserver [-force]
  '-force' option is provided but not used

=cut

sub configureIOS
{
   my $force = $_[0];
   my $run_as_owner = TRUE;
   my @output;
   my $status;

   if ($force eq "-force") {
     $status = srvctl($run_as_owner, "add ioserver -force");
   }
   else {
     $status = srvctl($run_as_owner, "add ioserver");
   }

   if ($status) {
     trace ("srvctl add ioserver ... success");
   } else {
     return FALSE;
   }

   return TRUE;
}



1;
