#!/usr/local/bin/perl
#
# $Header: has/install/crsconfig/oraafd.pm /st_has_12.2.0.1.0/1 2016/08/02 22:58:27 ssprasad Exp $
#
# oraafd.pm
#
# Copyright (c) 2008, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      oraafd.pm - Library module for ASM Filter Driver root install functions.
#
#    DESCRIPTION
#      oraafd.pm - Contains initial installation and deinstallation
#                   routines for AFD.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    ssprasad    07/19/16 - #23743975: AFD library check for installed query
#    ssprasad    03/24/16 - #22970859: Populate afd dskstr for Member cluster
#    muhe        03/09/16 - Fix bug 22894801: Cache and reuse the result of 
#                           'afddriverstate supported'
#    ssprasad    02/23/16 - #22689047: Support 'default' for AFD_CONF
#    ssprasad    01/14/16 - #22538718: Don't configure AFD in upgrade paths
#    ssprasad    08/20/15 - #21496397: Configure AFD if user indicated in OUI
#    gayavenk    08/05/15 - AFD Windows user fix
#    ssprasad    06/17/15 - #21153492:AFD not supported on Application Clusters
#    siyarlag    04/09/15 - use grid user for afd actions
#    ssprasad    03/09/15 - Change afd.conf to oracleafd.conf
#    siyarlag    02/06/15 - add ACTIONS to ora.driver.afd
#    ssprasad    11/11/14 - #19911021: invoke runAFDtool standalone
#    ssprasad    09/29/14 - Move isAFDSupported (crsutils.pm -> oraafd.pm)
#    xyuan       09/02/14 - rsc modeling
#    lcarvall    07/20/14 - Bug 13800615 - Use mnemonic string instead numeric values
#    apfwkr      04/04/14 - Backport ssprasad_bug-18389605 from main
#    ssprasad    04/01/14 - #18389605: handle failures in 'afdroot install'
#    ssprasad    03/11/14 - Fixes to support Windows AFD
#    ssprasad    03/04/14 - #18341155: Handle default str in kfod op=GPNPDSTR
#    ssprasad    02/21/14 - #18265466: Update ora.driver.afd ACL
#    ssprasad    02/20/14 - Add getASMDiscStrFromProfile
#    ssprasad    02/03/14 - Enable AFD for Solaris
#    ssprasad    01/23/14 - Fixes in actionAFDDriversResource() 
#    ssprasad    01/21/14 - Fix bug 18104036: Run 'runAFDtool' on all nodes
#                         - Fix bug 17494665: Record 'afdtool -rescan' failures
#    ssprasad    01/09/14 - Invoke runAFDtool only on non-first nodes
#    shmubeen    10/23/13 - return success in de-install case #17612647
#    siyarlag    10/08/13 - Bug17476998: Disable AFD installation on Exadata
#    shmubeen    09/19/13 - Fix for 17439522, run afdtool with ORACLE_OWNER
#                           user.
#    xyuan       08/22/13 - Remove compilation warnings
#    shmubeen    07/29/13 - Add an interface to add/start/stop afd resource
#    shmubeen    07/05/13 - fix for 17046459
#    shmubeen    07/01/13 - If Windows platform, just ignore, do not fail
#    shmubeen    05/17/13 - Remove additional check for ORA_ENABLE_AFD_INSTALL
#                           to enable afdinstallation
#    shmubeen    07/18/12 - Initial Creation
#

package oraafd;
use strict;
use English;
use File::Temp qw/ tempfile /;
use File::Spec::Functions;
use File::Find ();
use Term::ANSIColor;
use crsutils;
use s_crsutils;
use oraasm;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(installAFDriver
                   afdScanDevices
                   isAFDSupported
                   isAFDInstalled
                   removeAFDRoot
                   rm_afd_conf
                   afdPostConfigActions
                   actionAFDDriversResource);

push @EXPORT, @exp_func;

sub update_afd_conf
{
   my $user = $CFG->params('ORACLE_OWNER');
   my $grp = $CFG->params('ORA_ASM_GROUP');
     
   # Notes: Update afd.conf with the ASM discovery string from crsconfig params.
   # "afd_diskstring" is the key in oracleafd.conf.
   my ($afd_conf) = ''; 
   my ($afd_conf_12102) = '/etc/afd.conf'; 
   my (@lines);
   my ($i);
   my ($line);
      
   if ($CFG->platform_family eq 'windows') {
     my ($SYSTEM_ROOT) = $ENV{SYSTEMROOT};
     ($afd_conf) = "$SYSTEM_ROOT\\system32\\drivers\\oracleafd.conf";
   }
   else {
     ($afd_conf) = "/etc/oracleafd.conf";
   }

   # In 12.1.0.2.0, AFD was available only for Linux.
   # The AFD configuration file was named "/etc/afd.conf".
   # From 12.2, it's named "/etc/oracleafd.conf".
   # Customer could configure AFD after 12.1.0.2.0 was installed.
   # Hence, the file /etc/afd.conf may or may not be present.
   # If present, we just copy the configuration file and proceed ...
   # NOTE: In downgrade path (12.2 to 12.1.0.2.0), /etc/oracleafd.conf
   # will be removed. /etc/afd.conf is untouched.

   # Special handling for Linux in upgrade path
   if(($^O =~ /linux/i) && ($CFG->UPGRADE))
   {
     if(-e $afd_conf_12102)
     {
       trace("Found $afd_conf_12102. Indicates presence of ASMFD.");

       if (!copy_file($afd_conf_12102, $afd_conf)) {
         trace("Failed to copy $afd_conf_12102 to $afd_conf");
         return FAILED;
       }

       s_set_ownergroup($user, $grp, $afd_conf);
       s_set_perms ("0664", $afd_conf);
       return SUCCESS;
     }
   }

   #
   # In 12.2, if "/etc/oracleafd.conf" exists at this point,
   # it's a cleanup issue. Trace and proceed to create a new file.
   # TODO: For later releases, upgrade/downgrade needs a revisit.
   #       AFD configuration files have to be well handled.
   #
   if(-e $afd_conf)
   {
     trace("Unexpectedly found $afd_conf. The file Will be recreated.");
   }

   $lines[0] = "afd_diskstring='".getASMDiscStr()."'\n"; 
   trace("$afd_conf will be created with $lines[0]");

   # Write to afd.conf
   unless (open(FD,">$afd_conf"))
   {
     print_error(207, $afd_conf, $!);
     return FAILED;
   } 
   foreach $line (@lines)
   {
     print FD $line;
   }
   unless (close(FD))
   {
     print_error(272, $afd_conf, $!);
     return FAILED;
   } 

   s_set_ownergroup($user, $grp, $afd_conf);
   s_set_perms ("0664", $afd_conf);
}

sub rm_afd_conf
{
  my ($afd_conf) = ''; 

  if ($CFG->platform_family eq 'windows') {
    my ($SYSTEM_ROOT) = $ENV{SYSTEMROOT};
    ($afd_conf) = "$SYSTEM_ROOT\\system32\\drivers\\oracleafd.conf";
  }
  else {
    ($afd_conf) = "/etc/oracleafd.conf";
  }
    
  if(-e $afd_conf)
  {
    trace("Removing $afd_conf");
    unlink($afd_conf);
  }
  return SUCCESS;
}

sub installAFDriver
{
   my $afdroot;
   my $afdtool;
   my $ret = SUCCESS;
   my $afdInstall = FALSE;

   # If AFD not supported, then return success
   if (!isAFDSupported())
   {
      return $ret;
   }

   trace("installAFDriver");

   #
   # Update afd.conf with the ASM discovery string
   # "afd_diskstring" is the key in afd.conf
   #
   update_afd_conf();
    
   if ($CFG->platform_family eq 'windows') {
      $afdroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdroot.bat');
      $afdtool = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdtool.exe');
   }
   else {
      $afdroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdroot');
      $afdtool = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdtool');
   }

   if (-e $afdroot) {
      my $cmd = "$afdroot install";
      
      trace("Executing '$cmd'");
      my @output = system_cmd_capture($cmd);
      my $rc = shift @output;

      if ($rc == 0) {
        trace("AFD is installed successfully."); 
      }
      elsif ($rc != 2) {
         # 629 covers unload failures.
         # 650 covers load failures.
         #   (driver currently running, driver in use can't copy file,
         #    driver unable to load because Windows thinks its running
         #    but its not.)
         if ((scalar(grep(/629/, @output))) ||
             (scalar(grep(/650/, @output))) > 0)
         {
           if (is_dev_env())
           {
              trace ("Skipping reboot in development env ... success");
              return SUCCESS;
           }
           $ret = REBOOT;
         }
         else
         {
           $ret = FAILED;
           trace ("$cmd ... failed");
         }
      }
      # Output 2 is USM_NOT_SUPPORTED.
      # return true if AFD is not supported.
      else
      {
        $ret = SUCCESS;
        trace("AFD is not supported on this OS/architecture/kernel version.");
      }
   }
   else {
      trace("$afdroot not found");
      if (isAFDSupported())
      {
        # if afdroot not found and AFD supported, we have a problem
        # some required files are not here.
        trace("AFD is supported on this platform, but install files are missing.");
        $ret = FAILED;
      }
      else
      {
        # If afdroot not found and AFD not supported,
        # then assume everything is okay.
        trace("AFD is not supported on this platform.");
        $ret = SUCCESS;
      }
   }

  trace("ASM Filter driver install status is $ret");
  return $ret;
}

sub removeAFDRoot
#-------------------------------------------------------------------------------
# Function: remove AF driver
#-------------------------------------------------------------------------------
{
   my $chkpoints = $_;
   my $afdroot;
   my $ret = SUCCESS;  

   if ($CFG->platform_family eq 'windows') {
      $afdroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdroot.bat');
   }
   else {
      $afdroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdroot');
   }

   if (! (-e $afdroot)) {
      trace ("AFD is not configured");
      $ret = FAILED; 
      return $ret;
   }

   if ( $chkpoints == USECHKPOINTS )
   { 
     if (!isCkptexist("ROOTCRS_AFDUNINST")) 
     {
       trace("Writing checkpoint for ASM Filter driver uninstall");
       writeCkpt("ROOTCRS_AFDUNINST", CKPTSTART);
       $CFG->wipCkptName("ROOTCRS_AFDUNINST");
     }
   }

   if ( ($chkpoints == NOTUSECHKPOINTS) || !isCkptSuccess("ROOTCRS_AFDUNINST"))
   {
      $CFG->wipCkptName("ROOTCRS_AFDUNINST");

      my @cmd    = ($afdroot, 'uninstall');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("ASM Filter driver uninstall completed");
         if( $chkpoints == USECHKPOINTS )
         {
           writeCkpt("ROOTCRS_AFDUNINST", CKPTSUC);
         }
         $ret = SUCCESS;
      }
      elsif ( $rc != 2 )
      {
         if ((scalar(grep(/9118/, @output))) ||
             (scalar(grep(/9119/, @output))) ||
             (scalar(grep(/9348/, @output))) > 0)
         {
            trace("ASM Filter drivers unable to be uninstalled.");
            # Return success in de-install case. 
            if (($CFG->defined_param('HOME_TYPE')) || ($CFG->DEINSTALL))
            {
              trace ("Skipping reboot in deinstall case ... success");
              # We could not unload AF Drivers. Reboot the node after deinstall.
              print color 'bold';
              print_error(2506);
              print color 'reset';
                  
              if( $chkpoints == USECHKPOINTS )
              {
                writeCkpt("ROOTCRS_AFDUNINST", CKPTSUC);
              }
              return SUCCESS;
            }

            # We couldn't unload the old drivers or load the new ones.
            print color 'bold';
            print_error(2505);
            print color 'reset';
          }
          if ($chkpoints == USECHKPOINTS)
           {
              writeCkpt("ROOTCRS_AFDUNINST", CKPTFAIL);
           }
           exit 1;
      }
      else 
      {
         trace ("@cmd ... failed");
         if( $chkpoints == USECHKPOINTS )
         {
           writeCkpt("ROOTCRS_AFDUNINST", CKPTFAIL);
         }
         exit 1;
      }
   }
   return $ret;
}

=head2 isAFDSupported

   Determines if this platform is an AFD supported platform
   by calling 'afddriverstate supported'.

   In some specific setups, we consider AFD as not supported.
   Please see the notes below.

=head3 Parameters

   None

=head3 Returns

  TRUE  - AFD Supported
  FALSE - AFD Not Supported

=head3 Notes

  AFD is not configured in -->

  1/ Test Mode
  2/ Exadata
  3/ Application Cluster
  4/ Upgrade paths
  5/ unsupported platform/O.S 
  6/ User deselects the AFD option in OUI/responseFile
  7/ Member cluster where its Domain cluster (DSC) is not using AFD
  8/ Member cluster using indirect access (IOS)

=cut
sub isAFDSupported
{
   my $afdInstall   = FALSE; 
   my $loaded       = FALSE;
   my $libPresent   = FALSE;
   my $afdlib_12102 = '/opt/oracle/extapi/64/asm/orcl/1/libafd12.so';
   my $afd_option   = uc($CFG->params('AFD_CONF')); # true/false/default
   my $afddriverstate;
   my $AFD_supported = FALSE;
   my $oraasm        = $CFG->compASM;

   if (defined $CFG->AFDSupported)
   { 
      $AFD_supported = $CFG->AFDSupported;
      trace ("isAFDSupported: $AFD_supported");
      return $AFD_supported;
   }
 
   # 1/ Test Mode - (Old code. Not sure if it is still used)
   if (is_dev_env())
   {
      $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});

      # if this ENV is not set then we give up early
      if ( $afdInstall ne "TRUE" )
      {
         trace("AFD disabled because of ENV in test mode");
         goto CACHE_AFDSUPPORTED;
      }
   }

   # 2/ Exadata - (May not add value to have AFD)
   if (s_is_Exadata()) {
      trace("AFD is not supported on Exadata system");
      goto CACHE_AFDSUPPORTED;
   }

   # 3/ Application Cluster - (May not add value to have AFD)
   if (isAppCluster()) {
      trace("AFD is not supported on Application Cluster");
      goto CACHE_AFDSUPPORTED;
   }

   # 4/ In upgrade paths, don't configure AFD. Don't want
   # to force the existing 'labelled' disks to be managed by AFD.(asmlib & Win)
   # Except if AFD is currently in use.(which can be only on Linux w/ 12102)
   if($CFG->UPGRADE)
   {
     if (!($^O =~ /linux/i))
     {
       trace("AFD is not configured during upgrade.");
       goto CACHE_AFDSUPPORTED;
     }

     # Check if AFD driver is loaded
     $loaded = isAFDLoaded();
     if ($loaded) 
     {
       trace("Found AFD driver loaded.");
     }

     # Check if AFD library is present
     if (-e $afdlib_12102) 
     {
       $libPresent = TRUE;
       trace("Found AFD library: $afdlib_12102");
     }

     if (($libPresent) || ($loaded)) 
     {
       trace("Indicates presence of AFD");
     }
     else 
     {
       trace("AFD is not configured during upgrade");
       goto CACHE_AFDSUPPORTED;
     }
   }

   # check if AFD binaries are present
   if ($CFG->platform_family eq 'unix') 
   {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate');
   }
   elsif ($CFG->platform_family eq 'windows') 
   {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate.bat');
   }
   if (! (-e $afddriverstate)) 
   {
      trace ("$afddriverstate not found");
      goto CACHE_AFDSUPPORTED;
   }

   # 5/ check if AFD is supported on the given platform/O.S
   my @cmd = ($afddriverstate, "supported");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if ($rc != 0) 
   {
      trace ("AFD is not supported");
      goto CACHE_AFDSUPPORTED;
   }
   else 
   {
      trace ("AFD is supported");
   }

   # 6/ User deselected the AFD option in OUI/responseFile
   if ($afd_option eq "FALSE") 
   {
      trace("AFD Configuration is not selected");
      goto CACHE_AFDSUPPORTED;
   }

   ####
   # Member cluster related checks
   # NOTE: OUI sends 'default' for member clusters, Upgrades.
   # Pre 12.2 does not have member cluster. So, limiting the check
   # only to fresh installs.
   ####
   if ($afd_option eq "DEFAULT")
   {
     if (!($CFG->UPGRADE) && (isFarASM()))
     {
       ####
       # 7/ No AFD for Member cluster where it's DSC is not using AFD
       ####
       if ($oraasm->getvalFromManifestFile("AFD_STATE") eq 
           $oraasm->AFD_STATE_NOAFD) 
       {
         trace("AFD is not configured in ASM member Cluster " .
               "when the Domain Services Cluster is not using AFD");
         goto CACHE_AFDSUPPORTED;
       }

       ####
       # 8/ Don't configure AFD for member cluster using indirect access
       ####
       if ($oraasm->getvalFromManifestFile("ASM_ACCESS_MODE") eq
           $oraasm->ASM_DSKMODE_INDIRECT)
       {
         trace("AFD is not configured in ASM member Cluster " .
               "using indirect access");
         goto CACHE_AFDSUPPORTED;
       }
     }
   }

   $AFD_supported = TRUE;

CACHE_AFDSUPPORTED:
   $CFG->AFDSupported($AFD_supported);
   return $AFD_supported;
}

=head2 isAFDLibPresent

   Determines whether the AFD Library is present in system path

=head3 Parameters

=head3 Returns

  TRUE  - AFD Library is present in system path
  FALSE - AFD Library not present in system path

=head3 Notes


=cut
sub isAFDLibPresent
{
   my $present     = FALSE;
   my $SYSDRIVE    = $ENV{SYSTEMDRIVE};
   my $afdlib_path;

   if ($CFG->platform_family eq 'windows') {
      $afdlib_path = "$SYSDRIVE\\oracle\\extapi\\64\\asm\\oraafd12.dll";
   }
   else {
      $afdlib_path = "/opt/oracle/extapi/64/asm/orcl/1/libafd12.so";
   }

  if(-e "$afdlib_path")
  {
    trace ("AFD Library is present");
    $present = TRUE;
  }
  else
  {
    trace ("AFD Library is not present");
  }

  return $present;
}

=head2 isAFDInstalled

   Determines whether AFD is installed
     a/ Check for AFD driver presence
     b/ Check for AFD Library presence

=head3 Parameters

   Silent Option

=head3 Returns

  TRUE  - AFD Installed
  FALSE - AFD Not Installed

=head3 Notes


=cut
sub isAFDInstalled
{
   my $silent = shift;
   my $AFD_installed  = FALSE;
   my $afddrv_present = FALSE;
   my $afdlib_present = FALSE;
   my $afddriverstate;

   if ($CFG->platform_family eq 'windows') {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate.bat');
   }
   else {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate');
   }

   # check if afd is installed
   if (! (-e $afddriverstate)) {
      trace ("$afddriverstate not found");
      return FALSE;
   }

   my @cmd = ($afddriverstate, "installed");
   if ($silent)
   {
      push @cmd, '-s';     
   }
   trace("Running @cmd");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if ($rc == 0) {
      $afddrv_present = TRUE;
      trace ("AFD Driver is installed");
   }
   else {
      $afddrv_present = FALSE;
      trace ("AFD Driver is not installed");
   }

   # Check for AFD Library presence as well
   $afdlib_present = isAFDLibPresent();

  if (($afddrv_present) && ($afdlib_present))
  {
     $AFD_installed = TRUE;
     trace ("AFD is installed");
  }
  else {
      trace ("AFD is not installed");
   }

  return $AFD_installed;
}

sub LoadAFDriver
{
   my $chkpoints = $_;
   my $afdload;
   my $ret;

   if ($CFG->platform_family eq 'windows') {
      $afdload = catfile($CFG->ORA_CRS_HOME, 'bin', 'afdload.bat');
   }
   else {
      $afdload = catfile($CFG->ORA_CRS_HOME, 'bin', 'afdload');
   }

   if (! (-e $afdload)) {
      trace ("AFD is not configured");
      return;
   }

   if ( $chkpoints == USECHKPOINTS )
   {
     if (!isCkptexist("ROOTCRS_AFDLOAD"))
     {
        trace("Writing checkpoint for ASM Filter driver Load");
        writeCkpt("ROOTCRS_AFDLOAD", CKPTSTART);
        $CFG->wipCkptName("ROOTCRS_AFDLOAD");
     }
   } 

   if ( ($chkpoints == NOTUSECHKPOINTS ) || !isCkptSuccess("ROOTCRS_AFDLOAD"))
   {
      $CFG->wipCkptName("ROOTCRS_AFDLOAD");

      my @cmd    = ($afdload, 'start');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("ASM Filter driver Loading completed");
         if( $chkpoints == USECHKPOINTS )
         {
           writeCkpt("ROOTCRS_AFDLOAD", CKPTSUC);
         }  
      }
      else
      {
         trace ("@cmd ... failed");
         if( $chkpoints == USECHKPOINTS )
         {
           writeCkpt("ROOTCRS_AFDLOAD", CKPTFAIL);
         }
         exit 1;
      }
   }
}

sub unloadAFDriver
{
   my $afdload;
   my $ret = FAILED;

   if ($CFG->platform_family eq 'windows') {
      $afdload = catfile($CFG->ORA_CRS_HOME, 'bin', 'afdload.bat');
   }
   else {
      $afdload = catfile($CFG->ORA_CRS_HOME, 'bin', 'afdload');
   }

   # check if AFD is supported
   if (! (-e $afdload)) {
      trace ("$afdload not found");
      return FAILED;
   }

   my @cmd = ($afdload, "stop");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;
   
   if ($rc == '') {
      $ret = SUCCESS;
   }
   else {
      $ret = FAILED;
      trace ("AFD is not supported");
   }

  return $ret;
}

sub isAFDLoaded
{
   my $AFD_loaded = FALSE;
   my $afddriverstate;

   if ($CFG->platform_family eq 'windows') {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate.bat');
   }
   else {
      $afddriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'afddriverstate');
   }

   # check if AFD is supported
   if (! (-e $afddriverstate)) {
      trace ("$afddriverstate not found");
      return FALSE;
   }

   my @cmd = ($afddriverstate, "loaded");
   my @out = system_cmd_capture(@cmd);
   my $notloaded = grep (/false/i, @out);
   my $loaded = grep (/true/i, @out);

   if ($loaded eq 1) {
      $AFD_loaded = TRUE;
      trace ("AFD is loaded");
    }
   elsif($notloaded eq 1) {
      trace ("AFD is not loaded");
   }
   else
   {
     trace ("Error in executing $afddriverstate"); 
   }

  return $AFD_loaded;
}

sub actionAFD
{
 my $action = shift;
 my $ret = FAILED;
 my $retval = FALSE;

 if (isAFDSupported())
 {
   $retval = isAFDLoaded();
   if($retval eq 0)
   {
      return;
   } 
   if ($retval eq "loaded") 
   {
      if($action eq "stop")
      {
        $retval = unloadAFDriver();   
        if ($retval eq 1)
        {
          trace("AFD is unloaded"); 
          $ret = SUCCESS;
        } 
        else
        {
          trace("Error while unloading AFD");
        }   
      }
   }   
   elsif ($action eq "start")
   {
     loadAFD();  
   } 
 }
}

=head2 actionAFDDriversResource

   Create/Start/Delete the AFD Drivers Resource. 

=head3 Parameters

   $action - "stop"/"add"/"start"/"delete" actions for the drivers resource

=head3 Returns

  TRUE  - Creation/Start/Delete successful
  FALSE - Creation/Start/Delete failed

=head3 Notes

=cut

sub actionAFDDriversResource
{
  my ($asmowner, $user, $asmgrp, $action) = @_;
  my $crsctl  = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my $ret     = FAILED;

  if (isAFDSupported()) 
  { 
    my @out  = system_cmd_capture($crsctl, "stat", "res", "ora.driver.afd",
                                  "-init");
    my $notExists = grep (/CRS-2613/i, @out);
    my $status    = grep (/TARGET/i, @out);

    if (scalar($notExists) > 0 && 
        ($action eq "add" || $action eq "start"))
    {
      my @afd_attr = 
          ("ACL='owner:$asmowner:rwx,pgrp:$asmgrp:r-x,other::r--," .
           "user:$user:r-x'");

      # Handle Windows case where $user is NULL
      if ($CFG->platform_family eq "windows")
      {
        push @afd_attr, "ACTIONS='afdscan ".
                               "afdds ".
                               "afdtool ".
                               "afdstate'";
      } 
      else
      {
        push @afd_attr, "ACTIONS='afdscan,user:$user ".
                               "afdds,user:$user ".
                               "afdtool,user:$user ".
                               "afdstate,user:$user'";
      }
      $ret = crsctlResource("add", 
                            "ora.driver.afd", 
                            "ora.driver.afd.type", \@afd_attr);
    }
    elsif (scalar($status) > 0  && $action eq "delete")
    {
      $ret = crsctlResource($action, "ora.driver.afd");
    }
    elsif (scalar($status) > 0  && $action eq "stop")
    {
      # If the resource doesn't exist, then the catch all 'else' will
      # ensure that we return success.  Otherwise, we try and stop it.
      system_cmd_capture($crsctl, "stop", "res", "ora.driver.afd", "-init");
      $ret = SUCCESS;
    }
    else
    {
      # Make sure 'crsctl stat res' command ran properly
      if (scalar($status) > 0 || scalar($notExists) > 0) 
      {
        # The resource already exists or was already deleted  
        $ret = SUCCESS; 
      }
      else
      { 
        # There was an error running the command
        $ret = FAILED; 
      }
    }

    if ($action eq "start")
    {
      # Get the status again just to be sure  
      @out  = system_cmd_capture($crsctl, "stat", "res", "ora.driver.afd",
                                 "-init");
      my $statusOnline  = grep (/TARGET=ONLINE/i,  @out);
      my $statusOffline = grep (/TARGET=OFFLINE/i, @out);
      
      if (scalar($statusOnline) > 0) 
      {
        # The resource has been started already  
        $ret = SUCCESS; 
      }
      elsif (scalar($statusOffline) > 0 )
      { 
        # The resource has not been started 
        $ret = crsctlResource($action, "ora.driver.afd");
      }
      else
      {
        # The command failed or resource was not created
        $ret = FAILED;
      }
    }
  }
  else {
     # if AF Driver is not supported, nothing to do
     # We return SUCCESS to continue.
     $ret = SUCCESS;
  }
  return $ret;
}

=cut
=head2 afdScanDevices
   Runs 'asmcmd afd_scan'
=head3 Returns
  NONE
=head3 Notes
 This is run to discover the devices that are labeled. This enables other nodes
 to discover the disks that are labeled by node0.
 Under 'afdroot install', scan is already done. (afdboot -scandisk).
 Here, scan will additionally set the permissions of files under
 /dev/oracleafd/disks/ directory.
 Hence, the command is run as root.
=cut

sub afdScanDevices
{
   my $asmcmd = catfile ($CFG->ORA_CRS_HOME, 'bin', 'asmcmd');
   my $cmd    = "$asmcmd afd_scan"; 
   my @output;
   my $rc;

   trace("afdScanDevices Entry - ");
   trace("Executing '$cmd'");
   @output = system_cmd_capture($cmd);
   $rc = shift @output;
   trace("status: $rc output: @output \n");
   if($rc == 0)
   {
     trace("$cmd is run successfully.");
   }
   else
   {
     trace("$cmd has failed to run with status $rc");
     die(dieformat(180, $cmd));
   }

   return;  
}

=cut
=head2 getASMDiscStr
   Returns discovery string
=head3 Returns
  On Success, returns discovery string
=head3 Notes
 a/ This function needs a revisit for 12.2.0.2
 b/ Use of getASMDiscStrFromProfile is incorrect if lower version
    has AFD
 c/ For Member cluster, get it from Manifest file
=cut

sub getASMDiscStr
{
  my ($asm_discstr) = "";
  my $oraasm        = $CFG->compASM;

  # Bug-23000469 filed for OUI to fix DISKSTRING in member cluster.
  # Until it's fixed, for Member cluster will get the diskstring from
  # the manifest file (xml).
  #
  if (isFarASM())
  {
    $asm_discstr = $oraasm->getvalFromManifestFile("AFD_DISKSTRING");
  }
  else
  {
    # Get ASM_DISCOVERY_STRING from gpnp profile if it's an upgrade 
    # else get from crsconfig_params
    #
    if ($CFG->UPGRADE)
    {
      # Use kfod to query asm_diskstring
      # (which uses gpnp if available)
      #
      $asm_discstr = getASMDiscStrFromProfile();
    }
    else
    {
      $asm_discstr = $CFG->params('ASM_DISCOVERY_STRING');
    }
  }
  trace("ASM Diskstring: $asm_discstr");

  return ($asm_discstr);
}

=cut
=head2 getASMDiscStrFromProfile
   Runs 'kfod op=GPNPDSTR'
=head3 Returns

  SUCCESS  - returns asm_diskstring
  FAILED   - 'afdtool -rescan' failed

=head3 Notes
 Query asm_diskstring using kfod
=cut

#
#./kfod op=GPNPDSTR
#------------------------------------------
#Default Disk Discovery String
#=========================================
#/dev/rdsk/*
#

sub getASMDiscStrFromProfile
{
  trace("Using kfod to get ASM Diskstring");

  # if unable to query, return empty string
  my $asmdiskstr = "";

  my $kfod = catfile($CFG->ORA_CRS_HOME, 'bin', 'kfod');
  my @cmd = ($kfod, 'op=GPNPDSTR');
  my ($rc, @output) = system_cmd_capture(@cmd);
      
  if ($rc == 0)
  {
    foreach my $line (@output)
    {    
      trace($line);
      $asmdiskstr = $line;
    }
  }

  # kfod returns one line o/p for asm_diskstring.
  # Last line contains the asm_diskstring.
  #
  # kfod may return 0 but asm_diskstring may
  # be 'Not Set' or 'Not Available'.
  #
  # If asm_diskstring was never setup at profile creation
  # time or later, the default value in profile.xml is
  # "++no-value-at-profile-creation--never-updated-through-ASM++"
  # This special string also needs to be handled well.
  # For those, return empty string.
  #
  # Empty string is interpreted as default discovery string
  # by afdboot/afdtool.
  #
  if( ($asmdiskstr eq "Not Set") || 
      ($asmdiskstr eq "Not Available") || 
      ($asmdiskstr eq "++no-value-at-profile-creation--never-updated-through-ASM++") )
  {
    $asmdiskstr = "";
  }
  
  trace("ASM Diskstring: $asmdiskstr");

  return $asmdiskstr;
}

#------------------------------------------------------------------------------
# Function: Perform post config AFD actions
# 1/ Set AFD_DISKSTRING in OLR in fresh installs
#    (a/ In upgrade paths, we don't configure AFD)
#    (b/ If AFD is present in 12.1.0.2.0, AFD_DISKSTRING is already set)
#
# Args    : NONE
#
# Returns : NONE
#
# NOTES: a. Once oraafd.pm moves into new ROOT framework, this function
#           should be consumed and named appropriately.
#        b. ASMCA also sets AFD_DISKSTRING which is redundant.
#------------------------------------------------------------------------------
sub afdPostConfigActions
{
   my $asmcmd = catfile ($CFG->ORA_CRS_HOME, 'bin', 'asmcmd');
   my $afdstr = getASMDiscStr();
   my $cmd    = "$asmcmd afd_dsset \"$afdstr\"";
   my @output;
   my $rc;

   # If AFD not supported, then return success
   if (!isAFDSupported())
   {
      return;
   }

   if ($CFG->UPGRADE)
   {
     trace("AFD_DISKSTRING should already be set.");
     return;
   }

   trace("afdPostConfigureCurrentNode Entry - ");
   trace("Executing '$cmd'");
   $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@output, $cmd);
   trace("status: $rc output: @output \n");
   if($rc == 0)
   {
     trace("$cmd is run successfully.");
   }
   else
   {
     trace("$cmd has failed to run with status $rc");
     die(dieformat(180, $cmd));
   }

   return;
}

1;
