# 
# $Header: has/install/crsconfig/crscpcfg.pm /st_has_12.2.0.1.0/2 2016/08/10 23:28:24 xyuan Exp $
#
# crscpcfg.pm
# 
# Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      crscpcfg.pm - Implements the  actions needed for out of place GI home
#                    move to a new gold image
#
#    DESCRIPTION
#      
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    xyuan      08/10/16 - Fix bug 24438485
#    bbeelamk   08/03/16 - Fix bug 24301102
#    ksviswan   05/12/16 - do not copy addparams during gi move
#    muhe       04/26/16 - Fix bug 23173621
#    ksviswan   06/09/15 - correct copyConfigurationdata invoke
#    ksviswan   05/03/15 - Creation
# 

package crscpcfg;


use strict;
use English;
use Carp;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;

use crsutils;
use s_crsutils;
use crsgpnp;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);



use constant FIRST_NODE           => "first";
use constant LAST_NODE            => "last";

my @exp_func = qw( copyConfigurationData copyfiles );

push @EXPORT, @exp_func;


=head2 copyConfigurationData

  copies configuration files from source home to destination.

=head3 Parameters

  srcHome   - source home path from where to copy config data
  dstHome   - destination home path 

=head3 Returns

  SUCCESS  - all ok
  FAILED   - error occurred.

=cut


# This function is used to copy all required configuration files and dirs from
# the source (configured) GI home in case they are accidentally missing or
# the OOP patching using a gold image is performed.
sub copyConfigurationData
{
  my $srcHome = s_get_olr_file("crs_home");
  my $dstHome = $CFG->DESTCRSHOME;

  trace("Source home: $srcHome; Destination home: $dstHome");
  if ($srcHome ne $dstHome)
  {
    trace("Copying parameter files from src home to dst home ...");
    my $envtxt = 's_crsconfig_' . $CFG->HOST . '_env.txt';
    my @paramfiles = ('crsconfig_params',
                      's_crsconfig_defs',
                      'crsgenconfig_params',
                      $envtxt);

    foreach my $file (@paramfiles)
    {
      my $srcParaFile =
          catfile($srcHome, 'crs', 'install', $file);
      my $dstParaFile =
          catfile($dstHome, 'crs', 'install', $file);

      if (-e $srcParaFile)
      {
        my $needCopy = 0;

        if (! -e $dstParaFile)
        {
          trace("Copy needed as $dstParaFile is not present");
          $needCopy = 1;
        }
        elsif (! compareFiles($srcParaFile, $dstParaFile))
        {
          trace("Copy needed as $dstParaFile is different from $srcParaFile");
          $needCopy = 1;
          my $dstParaFileBak = $dstParaFile . '.oop-patching';
          copy_file($dstParaFile, $dstParaFileBak);
        }

        if (1 == $needCopy)
        {
          trace("Copying the parameter file $file ...");
          copy_file($srcParaFile, $dstParaFile);
        }
      }
    }

    crsutils->new(@_);    

    trace("Copying GPnP files from src home to dst home ...");
    copy_gpnpfiles($srcHome, $dstHome);

    copydir(catdir($srcHome,'gpnp','seed'), catdir($dstHome,'gpnp','seed'));

    trace("Copying configuration files/dirs from src home to dst home ...");
    copyfiles($srcHome, $dstHome);

    if (! $CFG->SIHA)
    {
      #Copy the dbs dir to propagate asm password files
      copydir(catdir($srcHome, 'dbs'), catdir($dstHome, 'dbs'));
    }

    my $cdataDir = catfile($dstHome, 'cdata');
    my $olrPath = ($CFG->SIHA) ? catfile($cdataDir, 'localhost') : $cdataDir;
    my $dstolrfile = catfile($olrPath,  $CFG->HOST . '.olr');
    my $srcolrfile = s_get_olr_file("olrconfig_loc");

    trace("Source OLR: $srcolrfile; Destination OLR: $dstolrfile");
    if ($srcolrfile ne $dstolrfile)
    {
      trace("Copying OLR data file from src to dst ...");
      copy_file($srcolrfile, $dstolrfile);
    }
  }
}

=head1 INTERNAL FUNCTIONS       
 
=head2 compareFiles 
 
  Compare the contents of two given files
 
=head3 Parameters       
 
  [0] File1
  [1] File2
                
=head3 Returns  
                        
  TRUE if two files have identical contents, otherwise FALSE

=cut

sub compareFiles
{
  my $file1 = $_[0];
  my $file2 = $_[1];
  my $rt = FALSE;

  trace("Comparing two files by content:  $file1 and $file2");

  unless (open(FILE1, "<$file1"))
  {
    trace("Could not open file $file1 for reading (error: $!)");
    return $rt;
  }

  unless (open(FILE2, "<$file2"))
  {
    trace("Could not open file $file2 for reading (error: $!)");
    return $rt;
  }

  my $file1Cnt = join("", <FILE1>);
  my $file2Cnt = join("", <FILE2>);

  close(FILE1);
  close(FILE2);

  if ($file1Cnt eq $file2Cnt)
  {
    trace("Two files have identical contents");
    $rt = TRUE;
  }
  else
  {
    trace("Two files have different contents");
  }

  return $rt;
}

sub copyfiles
{
   my $sourcedir = $_[0];
   my $destdir   = $_[1];
   my $host = tolower_host();

   # Copy network/admin/, cdata and opmn/conf files
   if (! $CFG->SIHA) {
      copydir(catdir($sourcedir,'network','admin'), catdir($destdir,'network','admin'));
      copydir(catdir($sourcedir,'cdata'), catdir($destdir,'cdata'));

      # for opmn/conf, we only need to copy ons.config file
      my $sourcefile = catfile ($sourcedir, 'opmn', 'conf', 'ons.config');
      my $destfile   = catfile ($destdir, 'opmn', 'conf', 'ons.config');
      copy_file ($sourcefile, $destfile,
                 $CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'));
   }
   else {
      copydir (catdir($sourcedir,'network','admin'), catdir($destdir,'network','admin'));

      #Remove line starting with ADR_BASE from listener.ora
      my $listenerfile = catfile($destdir,'network','admin','listener.ora');
      my $listenerfilebkup = catfile($destdir,'network','admin','listener.ora.bak');

      open (FP, "<","$listenerfile") || die(dieformat(206, $listenerfile, $!));

      my @file_lines = <FP>;
      close( FP );

      #backup the listener.ora file.
      if (!copy_file($listenerfile, $listenerfilebkup))
      {
         trace("Info: Failed to copy from $listenerfile to $listenerfilebkup");
      }

      #Write into listener.ora only those lines that don't begin with ADR_BASE

      open (FP, ">","$listenerfile") || die(dieformat(207, $listenerfile, $!));

      foreach my $line ( @file_lines ) {
         print FP $line unless ( $line =~ /^ADR_BASE/ );
      }
      close ( FP );
   }
}

