#!/usr/bin/perl

package module::DBUtilServices;
use warnings;
use strict;

use Exporter qw(import);
use File::stat;
use File::Copy;
use File::Path qw(mkpath rmtree);
use File::Find;
use File::Basename qw(dirname basename);
use Cwd  qw(abs_path);
use lib dirname(abs_path($0));

our @EXPORT_OK = qw(getActiveUser getFileOwner getFileOwnerGroup changeFileOwner changeFilePermission getJavaHomePath getJreMemoryOptions copyRecursively createDirectory removeDirectory openFile closeFile writeFile readFile removeFile findFilePattern);
use module::CommandOptions qw(getOptValue optExists);

#Getting current executing user
sub getActiveUser{
 my @_userDetails=getpwuid($<);
 return $_userDetails[0];
}

#Getting file owner name
sub getFileOwner{
 my $file = $_[0];
 my $userName = getpwuid(stat($file)->uid);
 return $userName;
}

#Getting file owner name
sub getFileOwnerGroup{
 my $file = $_[0];
 my $grpName = getgrgid(stat($file)->gid);
 return $grpName;
}

sub isWindows{
  my $windows=0;
  my $osname = $^O;
  if( $osname eq 'MSWin32' ){
	$windows = 1;
  }
  
  return $windows
}

sub getJavaHomePath{
  my $currentHome=$_[0];
  my $JAVA_HOME=$currentHome."/OPatch/jre";
  my $JAVA_REL_PATH="/bin/";
  
  if(isWindows){
   $JAVA_REL_PATH=$JAVA_REL_PATH."java.exe";
  }else{
	$JAVA_REL_PATH=$JAVA_REL_PATH."java";
  }
  
  my $JAVA=$JAVA_HOME.$JAVA_REL_PATH;
  if(! -e $JAVA){
   $JAVA_HOME=$currentHome."/jdk";
   $JAVA=$JAVA_HOME.$JAVA_REL_PATH;
   if(! -e $JAVA){
     print("\nJava could not be located. opatchauto cannot proceed!\n");
     print("opatchauto returns with error code = 2\n");
     exit(2);
   }
  }
 
  if(! -x $JAVA){
    print("\n$JAVA is not a valid executable for this platform. opatchauto cannot proceed!\n");
    print("opatchauto returns with error code = 2\n");
    exit(2);
  }
  if(isValidJavaVer($JAVA)==1){
   my $javaVerStr=`$JAVA -version 2>&1`;
   $javaVerStr =~ m/java version \"(.{3})/s;
   $javaVerStr=$1;
   print("\nUnsupported Java version $javaVerStr. opatchauto cannot proceed!\n");
   print("opatchauto returns with error code = 2\n");
   exit(2);
  }
   my $OPT_JRE=$_[1];
  if(length($OPT_JRE) > 0){
     $OPT_JRE=~ s!/*$!!g;
     $JAVA_HOME=~ s!/*$!!g;
     if($OPT_JRE ne $JAVA_HOME){
      print("warning: $OPT_JRE will not be used but the jre from $JAVA_HOME will be used. \n");
     }
  }
  return $JAVA_HOME;
}

#Get supplied jre version
sub getJreVersion{
  my $retVal=1;
  my $javaVerStr=`$_[0] -version 2>&1`;
  my $javaVer="";
 if ($javaVerStr =~ m/java version \"(.{3})/s ) {
   $javaVer=$1;
   $javaVer=~s/\.//;
 }
  return $javaVer;
}

#Checking java version is greater than 1.7
sub isValidJavaVer{
  my $javaVersion=getJreVersion($_[0]);
  my $retVal=0;
  if ($javaVersion < 17 ) {
    $retVal=1;
  }
   return $retVal;
}

sub getJreMemoryOptions{
  my $oraParamIniFile=$_[0];
  my $jreMemoryOpts=$_[1];
  if(length $jreMemoryOpts <= 0){
      $jreMemoryOpts=extractJreMemoryOpts($oraParamIniFile);
      $jreMemoryOpts=updateJreMemoryOptsForJRockit($jreMemoryOpts);
   }else{
    if(! $jreMemoryOpts =~ m/ \-d64/){
       $jreMemoryOpts=extractJreMemoryOpts($oraParamIniFile);
       if(! $jreMemoryOpts =~ m/ \-d64/){
        $jreMemoryOpts=$jreMemoryOpts." -d64";
       }
    }
   }
   
   my $JAVA=$_[2];
   my $jreVersion=getJreVersion($JAVA);
   
   if($jreVersion >= 18){
    $jreMemoryOpts=~s/MaxPermSize/MaxMetaspaceSize/g;
   }
   return $jreMemoryOpts;
}

sub extractJreMemoryOpts{
  my $oraParamIniFile=$_[0];
  my $jreMemoryOpts="";
 if (-e $oraParamIniFile){
   open my $fh, '<', $oraParamIniFile or die "Could not open '$oraParamIniFile' $!\n";
    while (my $line = <$fh>){
     if($line=~s/JRE_MEMORY_OPTIONS=//){
        $jreMemoryOpts=$line;
        $jreMemoryOpts=~s/"//g;
        last;
     }
    }
    close $fh;
 }
 chomp $jreMemoryOpts;
 return $jreMemoryOpts;
}


sub updateJreMemoryOptsForJRockit{
 my $jreMemoryOpts=$_[0];
 if(defined $ENV{'JDK'}){
     my $JDK=$ENV{'JDK'};
     if($JDK =~ m/jrockit/){
       $jreMemoryOpts=~s/XX:MaxPermSize=/ms/;
     }
  }
  return $jreMemoryOpts;
}

sub changeFileOwner{
  my $userName=$_[0];
  my $grpName=$_[1];
  my $path=$_[2];
  my $uid=getpwnam($userName);
  my $gid=getgrnam($grpName);
 if (-d $path) {
   find(sub{chown($uid, $gid,$_)}, $path);
  }else{
    chown($uid, $gid, "$path");
  }
  return 1;
}

sub findFilePattern{
 my $sourcedir=$_[0];
 my $filePattern=$_[1];
 my @filelist;
 find(sub { push @filelist, $File::Find::name }, $sourcedir);
  foreach my $filepath (@filelist){
     if ( $filepath =~ /$filePattern/){
       return 1;
     }
   }
   
   return 0;
}

sub changeFilePermission{
  my $path=$_[0];
  my $permission=$_[1];
  if (-d $path) {
   find(sub{chmod(oct($permission),$_)}, $path);
  }else{
    chmod($permission, "$path");
  }
  return 0;
}

sub copyRecursively{
  my $sourcedir = $_[0];
  my $destdir   = $_[1];
   my $filepath;
   my $filename;
   my @filelist;
   my $destfilepath;
   my $dirpath;
   find(sub { push @filelist, $File::Find::name }, $sourcedir);
   
   foreach $filepath (@filelist)
   {
     if ( -f $filepath)
     {
        $filename = basename($filepath);
        $dirpath = dirname($filepath);
        $dirpath =~ s/$sourcedir/$destdir/g;
        if( !-e $dirpath){
         createDirectory($dirpath);
        }
        $destfilepath = "$dirpath/$filename";
        copy($filepath, $destfilepath);
     }
   }
}


sub createDirectory{
  my $dir=$_[0];
  if(!-e $dir){
    eval { mkpath($dir) };
    if ($@) {
      print "Couldn't create $dir: $@";
      return 1;
    }
  }
  return 0;
}

sub removeDirectory{
  my $dir=$_[0];
  if(-e $dir){
    eval { rmtree($dir) };
    if ($@) {
      print "Couldn't remove $dir: $@";
      return 0;
    }
  }
  return 1;
}

sub removeFile{
 my $file = $_[0];
 if(-e $file) {
  unlink $file;
 }
 
 return 1;
}

sub openFile{
 my $mode=$_[1];
 my $file=$_[0];
 my $FileHandler;
 if($mode == 1){
   # Use the open() function to create the file.
   if(!open $FileHandler, ">".$file){
    print("\n perl say $! \n");
   }
}else{
   # Use the open() function to create the file.
  open $FileHandler,$file;
}
  return $FileHandler;
}


sub closeFile{
 my $FileHandler=$_[0];
 # close the file.
 close $FileHandler;
}

sub writeFile{
 my $fh=$_[0];
 my $content= $_[1];
 # Write some text to the file.
 print $fh "$content\n";
}

sub readFile{
 my $fh=$_[0];
 # Read the file one line at a time.
 my $line = <$fh>;
 return $line; 
}