/*
 * $Header: oals.h 05-feb-2001.15:08:01 dthakuri Exp $ 
 */

/* Copyright (c) 1991 by Oracle Corporation */
/*
   NAME
     oals.h - Office Automation Language Support
   DESCRIPTION
     NLS support functions. 

     These functions fill some of the holes in functionality in the 
     NLS runtime library.

   PUBLIC FUNCTION(S)
      oalstol - OA Language String TO Long
      oalsubx - OA Language SUBstring eXtract
      oalsstr - OA Language Search for subSTRing 
      oalsceq - OA Language String Compare for EQuality 
      oalscfc - OA Language String Compare Future Character
      oalsbgc - OA Language Single-Byte Get Character
      oalschq - OA Language Support CHaracter eQuals 
      oalsadv - OA Language Support ADVance
   PRIVATE FUNCTION(S)
   RETURNS
   NOTES
     Because these functions often call the more-specific NLSRTL
     functions, your code may be more efficient if you used those
     functions instead of these. 
   MODIFIED   (MM/DD/YY)
    dthakuri   02/05/01  - Olint error fix.
    ewinner    10/07/96 -  core3 upgrade 
    tkukulsk   04/13/94 -  fix synatax error, add macro to header 
    tkukulsk   04/11/94 -  add some useful macros 
    tkukulsk   04/11/94 -  add OALS_NONE to public constants 
    tkukulsk   03/17/94 -  add prototype for oalsadv 
    tkukulsk   02/22/94 -  Creation 
*/



#ifndef OALS
#define OALS

# ifndef S
# include <s.h>
# endif

# ifndef LX
# include <lx.h>
# endif  /* LX */

/* PUBLIC TYPES AND CONSTANTS */

/* the following bit patterns are used to indicate options to
   NLS functions */

/* no special equivalence is to be used */
#define OALS_NONE 0

/* single-byte and multi-byte are equivalent */
#define OALS_SM 1

/* comparison should be done case-insensitive */
#define OALS_UL 2

/* copy to single-byte */
#define OALS_SB 4

/* copy to multibyte */
#define OALS_MB 8

/* copy to uppercase */
#define OALS_UC 16

/* copy to lowercase */
#define OALS_LC 32

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
uword oalstol (/*_ listp *, sb4 *, lxglo * _*/);
#endif

/*
*  NAME
*
*  oalstol - OA Language String TO Long
*  
*
*  USE
*
*  uword oalstol ( listp *reader, slong *result, lxglo *glo );
*
*
*  DESCRIPTION
*  
*  This reads an integer value (Base 10) from an open string.
*
*  Leading whitespace, and the number itself  will be consumed, if a
*  value is returned.
*
*  
*  PARAMETERS
*
*  listp  *reader - (IN/OUT) Handle to a string opened for reading.
*
*  slong  *result - (OUT) Pointer to a signed long integer. If a value is
*                         succesfully read, it will be placed here.
*
*  lxglo  *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  uword   <rval> - Number of characters (not neccisarily bytes) consumed.
*
*  
*  NOTES
*  
*  A minus-sign may be placed before the first digit in order to input
*  a negative number.
*
*  This does not distinguish between single-byte and multibyte text
*
*  If no number is read, or if the value overflows, <reader> and
* <result> remain unchanged, and zero is returned
*
*  If the number exceeds SB4MAXVAL, the entire number is consumed, and
*  SB4MAXVAL is returned. (The negative case also holds)
*    
*/
size_t oalsubx(/*_ text *dest, text *source, uword start, uword len,
                 uword flags, lx_langid lid, lxglo *glo _*/);
/*
*  NAME
*
*  oalsubx - OA Language SUBstring eXtract
*
*
*  USE
*
*  size_t oalsubx ( text *dest, text *source, uword start, uword len,
*                   uword flags, lx_langid lid, lxglo *glo );
*
*  
*  DESCRIPTION
*
*  This copies a substring from <source> to <dest>.
*
*  The initial <start> characters of <source> are skipped, and then
*  <len> additional characters are copied, providing that end-of-string
*  is not encountered. <Dest> is then NULL-terminated. 
*
*  The number of bytes written to DEST (including NULL-termination)
*  is returned.
*  
*  
*  PARAMETERS
*
*  text      *dest   - (OUT) Pointer to destination buffer.
*
*  text      *source - (IN)  Pointer to source buffer.
*
*  uword      start  - Number of characters to skip before copying.
*
*  uword      len    - Maximum number of characters to copy.
*
*  uword      flags  - how should this substring be copied?
*                    -- OALS_SB convert to single-byte
*                    -- OALS_MB convert to multi-byte
*                    -- OALS_UC convert to upper-case
*                    -- OALS_LC convert to lower-case
*                    - ( X-byte flags may be combined with X-case
*                        flags, but performance will be reduced)
*
*  lx_langid  lid    - language ID for use by NLSRTL
*  
*  lxglo     *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  size_t    <rval>  - Number of bytes requred to store output string.
*
*  
*  NOTES
*  
*  If you already have a handle to the beginning of the substring, it
*  will be slightly more efficient to use inline code top copy the substring.
*
*  If you specify that you want to convert both the CASE and the
*  SM-byte-ness of a string, performance will be greatly reduced.
*  INSTEAD, if you have an extra buffer, do one conversion and then
*  the other. 
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
sword oalsstr (/*_ listp *reader, text *pat, uword flags, lxglo *glo _*/);
#endif

/*
*  NAME
*
*  oalsstr - OA Language Search for subSTRing 
*
*
*  USE
*
*  sword oalsstr ( listp *reader, text *pat, uword flags, lxglo *glo );
*
*  DESCRIPTION
*
*  Finds first occurence of <pat> within <reader>, advances
*  <reader> to the beginning of the matching string, and returns the
*  number of characters skipped. If no match is found, <reader> will be
*  unchanged, and a value of -1 will be returned.
*
*  <Flags> is used to indicate various types of equivalency (see parameters)
*  
*  PARAMETERS

*
*  listp *reader - (IN/OUT) Handle to a string opened for reading.
*
*  text  *pat    - (IN) Pointer to the string to be searched for in <reader> 
*
*  uword  flags  - Bit field, specifying equvalency.
*                -- OALS_SM   accept "<A>" as equivalent to "A"
*                -- OALS_UL   accept "a"   as equivalent to "A"
*                modes may be OR-ed together
*                  
*  lxglo *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  uword  <rval> - Number of characters (not neccisarily bytes) consumed.
*
*  
*  NOTES
*    
*  performance may drop off quickly when uppercase/lowercase equivalency
*  is turned on, because of the increased number of function calls
*    
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
uword oalschq (/*_ listp *a, listp *b, uword flags, lxglo *glo _*/);
#endif

/*
*  NAME
*
*  oalschq - OA Language Support CHaracter eQuals 
*
*
*  USE
*
*  uword oalschq ( listp *a, listp *b, uword flags, lxglo *glo );
*
*  DESCRIPTION
*
*  returns TRUE if the current characters of <a> and <b> are equivalent 
*
*  <Flags> is used to indicate various types of equivalency (see parameters)
*  
*  PARAMETERS

*
*  listp *a      - (IN) Handle to a string opened for reading.
*
*  listp *b      - (IN) Handle to a string opened for reading.
*
*  uword  flags  - Bit field, specifying equvalency.
*                -- OALS_SM   accept "<A>" as equivalent to "A"
*                -- OALS_UL   accept "a"   as equivalent to "A"
*                modes may be OR-ed together
*                  
*  lxglo *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  uword  <rval> - Number of characters (not neccisarily bytes) consumed.
*
*  
*  NOTES
*    
*  Performance may drop off quickly when doing uppercase/lowercase
*  equivalence, due to the increased number of function calls required.
*    
*/
uword oalsceq (/*_ text *a,text *b, uword flags, lx_langid lid,lxglo *glo_*/);
/*
*  NAME
*
*  oalsceq - OA Language String Compare for EQuality 
*
*
*  USE
*
*  uword oalsceq ( text *a,text *b,uword flags, lx_langid lid, lxglo *glo );
*
*  DESCRIPTION
*
*  Finds first occurence of <pat> within <reader>, advances
*  <reader> to the beginning of the matching string, and returns the
*  number of characters skipped. If no match is found, <reader> will be
*  unchanged, and a value of -1 will be returned.
*
*  <Flags> is used to indicate various types of equivalency (see parameters)
*  
*  PARAMETERS
*
*  text      *a      - (IN) Pointer to a closed native string
* 
*  text      *b      - (IN) Pointer to a closed native string
*
*  uword      flags  - Bit field, specifying equvalency desired
*                    -- OALS_SM   accept "<A>" as equivalent to "A"
*                    -- OALS_UL   accept "a"   as equivalent to "A"
*                    modes may be OR-ed together
*                  
*  lx_langid  lid    - language ID for use by NLSRTL
*  
*  lxglo     *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  uword      <rval> - TRUE if <a> and <b> are equivalent as specified
*                      in <flags>
*
*  NOTES
*    
*  This function checks for equivalence. It does not compare strings.
*  Because of this, it cannot be used for sorting, but may be faster
*  than string comparison.
*
*    
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
uword oalscfc (/*_ listp *reader, uword nchars , text c , uword flags,
                  lxglo *glo _*/);
#endif

/*
*  NAME
*
*  oalscfc - OA Language String Compare Future Character
*
*
*  USE
*
*  uword oalscfc ( listp *reader, uword nchars , text c , uword flags,
*                  lx_langid lid, lxglo *glo );
*
*  DESCRIPTION
*
*  Skips ahead <nchars> within <reader> and compares that character
*  with <c>. The comparison is specified by <flags>. This is merely a
*  comparison for equality. This cannot be used for sorting.
*
*  <Flags> is used to indicate various types of equivalency (see parameters)
*  
*  PARAMETERS
*
*  listp     *reader - (IN) String handle opened for reading
*
*  uword      nchars - number of characters to skip before comparison
* 
*  text       c      - Single-byte character to be compared against
*                      the future character in <reader>
*
*  uword      flags  - Bit field, specifying equvalency desired
*                    -- OALS_SM   accept "<A>" as equivalent to "A"
*                    -- OALS_UL   accept "a"   as equivalent to "A"
*                    modes may be OR-ed together
*                  
*  lx_langid  lid    - language ID for use by NLSRTL
*  
*  lxglo     *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  uword      <rval> - TRUE if <a> and <b> are equivalent as specified
*                      in <flags>
*
*  NOTES
*    
*  This function checks for equivalence. It does not compare strings.
*  Because of this, it cannot be used for sorting.
*    
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
text  oalsbgc (/*_ listp *reader, uword flags, lxglo *glo _*/);
#endif

/*
*  NAME
*
*  oalsbgc - OA Language Single-Byte Get Character
*
*
*  USE
*
*  text oalsbgc ( listp *reader, uword flags, lxglo *glo );
*
*  DESCRIPTION
*
*  returns ascii value of character, if listp is pointing to an ascii
*  character. If listp is pointing to a multi-byte equivalent, the
*  return value also depends upon <FLAGS>
*
*  <Flags> is used to indicate various types of equivalency (see parameters)
*  
*  PARAMETERS
*
*  listp     *reader - (IN) String handle opened for reading
*
*  uword      flags  - Bit field, specifying equvalency desired
*                    -- OALS_SM   accept "<A>" as equivalent to "A"
*                    -- (if OALS_SM isn't set will return 0 for <A>)
*                  
*  lxglo     *glo    - (IN/OUT) Pointer to NLSRTL global scratch.
*
*  text      <rval>  - 0 if character was not ascii
*                      ascii index of character, if ascii character
*
*  NOTES
*    
*  because this function returns '\0' for non-ascii characters, it
*  cannot be used to check for end-of-string
*
*  instead of calling this with OALS_SM, I suggest calling limtos
*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
uword oalsadv (/*_ listp *reader, uword nchars, lxglo *lig _*/);
#endif

/*
*  NAME
*
*  oalsadv - OA Language Support ADVance
*
*
*  USE
*
*  uword oalsadv ( listp *reader, uword nchars );
*
*  DESCRIPTION
*
*  Skips ahead <nchars> within <reader>, if possible.
*  Returns the number of characters skipped.
*
*  
*  PARAMETERS
*
*  listp     *reader - (IN/OUT) String handle opened for reading
*
*  uword      nchars - number of characters to skip
*  
*  uword      <rval> - number of characters actually skipped
*
*  NOTES
*    
*/


/*======================================================================
*
*  NAME
*  
*  OALSCPR - CoPy Remaining characters
*
*  USE
*  
*  void oalscpr ( listp *writer, listp *readerr, lxglo *glo );
*
*  listp *reader - (IN/OUT) null-terminated string handle opened for reading
*  
*  listp *writer - (IN/OUT) string handle opened for writing
*  
*  lxglo *glo    - (IN/OUT) LI global scratch memory
*  
*  
*  DESCRIPTION
*
*  the remaining characters of <reader> are copied to <writer>
*  
*  #define oalscpr(w,r,glo)
*   do {
*        while (!limnull(r))
*           {
*               DISCARD limcpc(w, r, glo);
*           }
*      } while (FALSE)
*
*
*   the do {...} while (FALSE) construct is a way of ensuring true
*   function-like behavior to the macro.
*======================================================================*/

/*
   the following two function's definition is nowhere to be found
   neither has it been invoked anywhere.  That's why, commenting
   it out in order to take care of Olint errors.
*/
#ifdef NEVER
void OALSCPR( listp *writer, listp*reader, lxglo *glo);
#endif

#define OALSCPR(writer,reader,glo) do { while (!lxmnull((reader), glo)) { \
DISCARD lxmcpc((writer),(reader),(glo)); } } while (FALSE)


#endif /* OALS */
